//=============================================================================
// RPGツクールMZ - StoreAbility.js
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc 能力値を保持するプラグインコマンドを定義します。
 * @author emoriiin979
 *
 * @help StoreAbility.js
 *
 * このプラグインは、能力を保持するためのプラグインコマンドを定義します。
 *
 * プラグインコマンド：
 *   能力値保存
 *     ・アクターの現時点の能力値を保持します。
 *
 *   能力値戻し
 *     ・保存してあった能力値をアクターに戻します。
 *       能力値を保存していなかった場合は何もしません。
 *
 *   能力値の絶対値変更
 *     ・能力値変更を絶対値で指定できます。
 *
 * @command store
 * @text 保存
 * @desc アクターの能力値を保存します。
 *
 * @arg variableId
 * @text 変数ID
 * @desc 能力値を格納する変数を指定します
 * @type variable
 *
 * @command restore
 * @text 戻す
 * @desc 保存済みの能力値を戻します
 *
 * @arg variableId
 * @text 変数ID
 * @desc 能力値を格納していた変数を指定します
 * @type variable
 *
 * @command absoluteChange
 * @text 能力値の絶対値変更
 * @desc 能力値変更を絶対値で指定できます
 *
 * @arg mhp
 * @text 最大HP
 * @desc 変更先の最大HPを指定してください
 * @type number
 *
 * @arg mmp
 * @text 最大MP
 * @desc 変更先の最大MPを指定してください
 * @type number
 *
 * @arg atk
 * @text 攻撃力
 * @desc 変更先の攻撃力を指定してください
 * @type number
 *
 * @arg def
 * @text 防御力
 * @desc 変更先の防御力を指定してください
 * @type number
 */

(() => {
    "use strict";

    const pluginName = "StoreAbility";

    //=========================================================================
    // プラグインコマンド
    //=========================================================================

    // 保存
    PluginManager.registerCommand(pluginName, "store", args => {
        const actor = $gameActors.actor(1);
        $gameVariables.setValue(args["variableId"], {
            mhp: actor.mhp,
            mmp: actor.mmp,
            atk: actor.atk,
            def: actor.def,
        });
    });

    // 戻す
    PluginManager.registerCommand(pluginName, "restore", args => {
        const obj = $gameVariables.value(args["variableId"]);
        if (obj) {
            const actor = $gameActors.actor(1);
            actor.addParam(0, obj.mhp - actor.mhp);
            actor.addParam(1, obj.mmp - actor.mmp);
            actor.addParam(2, obj.atk - actor.atk);
            actor.addParam(3, obj.def - actor.def);
        }
    });

    // 能力値の絶対値変更
    PluginManager.registerCommand(pluginName, "absoluteChange", args => {
        const actor = $gameActors.actor(1);
        actor.addParam(0, args["mhp"] - actor.mhp);
        actor.addParam(1, args["mmp"] - actor.mmp);
        actor.addParam(2, args["atk"] - actor.atk);
        actor.addParam(3, args["def"] - actor.def);
    });
})();
